#define D3D_DEBUG_INFO
#define DIRECTINPUT_VERSION 0x0800

#include <Windows.h>
#include <WindowsX.h>
#include <d3d9.h>
#include <d3dx9.h>

#pragma comment(lib, "d3d9")
#pragma comment(lib, "d3dx9")

#include "JewelBoard.h"

// ----------------------------------------

struct Configuration
{
    UINT       adapter;
    D3DDEVTYPE devType;
    D3DFORMAT  adapterFormat;

    D3DFORMAT spriteTextureFormat;
    D3DFORMAT backgroundTextureFormat;
    D3DFORMAT targetFormat;
};

Configuration g_Config;

D3DFORMAT FindDeviceFormat(IDirect3D9* pD3D, DWORD usage, D3DRESOURCETYPE resType, D3DFORMAT const* fmtList, DWORD numFormats)
{
    for (DWORD i = 0; i < numFormats; ++i)
    {
        D3DFORMAT const format = fmtList[i];

        if (D3D_OK == pD3D->CheckDeviceFormat(g_Config.adapter, g_Config.devType, g_Config.adapterFormat, usage, resType, format))
        {
            return format;
        }
    }
    return D3DFMT_UNKNOWN;
}

template < DWORD numFormats >
__forceinline
D3DFORMAT FindDeviceFormat(IDirect3D9* pD3D, DWORD usage, D3DRESOURCETYPE resType, D3DFORMAT const (&fmtList)[numFormats])
{
    return FindDeviceFormat(pD3D, usage, resType, fmtList, numFormats);
}

D3DFORMAT FindBackBufferFormat(IDirect3D9* pD3D, D3DFORMAT const* fmtList, DWORD numFormats)
{
    for (DWORD i = 0; i < numFormats; ++i)
    {
        D3DFORMAT const format = fmtList[i];

        if (D3D_OK == pD3D->CheckDeviceType(g_Config.adapter, g_Config.devType, g_Config.adapterFormat, format, TRUE))
        {
            return format;
        }
    }
    return D3DFMT_UNKNOWN;
}

template < DWORD numFormats >
__forceinline
D3DFORMAT FindBackBufferFormat(IDirect3D9* pD3D, D3DFORMAT const (&fmtList)[numFormats])
{
    return FindBackBufferFormat(pD3D, fmtList, numFormats);
}

IDirect3DTexture9* NewTexture(IDirect3DDevice9* pDevice, char const* fname, D3DFORMAT format)
{
    IDirect3DTexture9* texture = NULL;

    HRESULT const hr = D3DXCreateTextureFromFileExA(
        pDevice, fname,
        D3DX_DEFAULT_NONPOW2, D3DX_DEFAULT_NONPOW2, 0,
        0, format, D3DPOOL_MANAGED,
        D3DX_FILTER_POINT, D3DX_DEFAULT,
        0, NULL, NULL, &texture);

    if (FAILED(hr))
    {
        throw "Failed to load texture";
    }

    return texture;
}

enum GameState
{
    Animating,
    NoneSelected,
    FirstSelected,
    SwappingJewels,
    SwappingJewelsBack,
};

GameState g_State = Animating;
size_t g_FirstSelectedX = 0;
size_t g_FirstSelectedY = 0;
size_t g_SecondSelectedX = 0;
size_t g_SecondSelectedY = 0;

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch (message)
	{
	case WM_PAINT:
		hdc = BeginPaint(hWnd, &ps);
		EndPaint(hWnd, &ps);
		break;
    case WM_KEYDOWN: 
        switch (wParam) 
        { 
            case VK_ESCAPE:
                PostQuitMessage(0);
                return 0;
        }
		return DefWindowProc(hWnd, message, wParam, lParam);
    case WM_LBUTTONUP:
        {
            size_t x = GET_X_LPARAM(lParam);
            size_t y = GET_Y_LPARAM(lParam);

            RECT rect;
            GetClientRect(hWnd, &rect);

            float fx = float(x) / float(rect.right);
            float fy = float(y) / float(rect.bottom);

            if (fx > 320.0f / 800.0f)
            {
                fx = (fx * 800.0f - 320.0f) / 480.0f;

                // Jewel coordinates.

                x = size_t(floor(fx * 10.0f));
                y = size_t(floor(fy * 10.0f));

                if (g_State == NoneSelected)
                {
                    g_FirstSelectedX = x;
                    g_FirstSelectedY = y;
                    g_State = FirstSelected;
                }
                else if (g_State == FirstSelected)
                {
                    if (g_FirstSelectedX == x && g_FirstSelectedY == y)
                    {
                        g_State = NoneSelected;
                    }

                    if (g_FirstSelectedX == x && g_FirstSelectedY == y + 1
                     || g_FirstSelectedX == x && g_FirstSelectedY + 1 == y
                     || g_FirstSelectedX == x + 1 && g_FirstSelectedY == y
                     || g_FirstSelectedX + 1 == x && g_FirstSelectedY == y)
                    {
                        g_SecondSelectedX = x;
                        g_SecondSelectedY = y;
                        g_State = SwappingJewels;
                    }
                    else
                    {
                        g_FirstSelectedX = x;
                        g_FirstSelectedY = y;
                    }
                }
            }
        }
		break;
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return 0;
}

void DrawRotatedSpriteRectangle (IDirect3DDevice9* pDevice, float x0, float y0, float x1, float y1, float a, IDirect3DTexture9* pTexture);
void DrawSpriteRectangle        (IDirect3DDevice9* pDevice, float x0, float y0, float x1, float y1, IDirect3DTexture9* pTexture);
void DrawTexturedOpaqueRectangle(IDirect3DDevice9* pDevice, float x0, float y0, float x1, float y1, IDirect3DTexture9* pTexture);

int MyWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nCmdShow)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    // Initialization: create a window.
    // This part has nothing to do with D3D, but it is necessary nevertheless

    {
	    WNDCLASSEX wcex;

	    wcex.cbSize = sizeof(WNDCLASSEX);

	    wcex.style			= CS_HREDRAW | CS_VREDRAW;
	    wcex.lpfnWndProc	= WndProc;
	    wcex.cbClsExtra		= 0;
	    wcex.cbWndExtra		= 0;
	    wcex.hInstance		= hInstance;
	    wcex.hIcon			= LoadIcon(NULL, IDI_APPLICATION);
	    wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	    wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);
	    wcex.lpszMenuName	= NULL;
	    wcex.lpszClassName	= L"Bejeweled";
	    wcex.hIconSm		= LoadIcon(NULL, IDI_APPLICATION);

	    RegisterClassEx(&wcex);
    }

    DWORD const width  = 800;
    DWORD const height = 480;
    //DWORD const windowStyle = WS_OVERLAPPEDWINDOW;
    DWORD const windowStyle = WS_POPUPWINDOW | WS_CAPTION | WS_MAXIMIZEBOX;

    DWORD windowWidth;
    DWORD windowHeight;

    {
        RECT rect = { 0, 0, width, height };
        AdjustWindowRect(&rect, windowStyle, FALSE);
        windowWidth = rect.right - rect.left;
        windowHeight = rect.bottom - rect.top;
    }

    HWND const hWnd = CreateWindow(
        L"Bejeweled", L"Bejeweled",
        windowStyle,
        CW_USEDEFAULT, CW_USEDEFAULT, windowWidth, windowHeight,
        NULL, NULL, hInstance, NULL);

    if (!hWnd)
    {
        throw "Failed to create the window";
    }

    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);

    // Initialization: Initialize the D3D device and figure out its capabilities.

    D3DPRESENT_PARAMETERS pp;

    ZeroMemory(&pp, sizeof(pp));
    pp.BackBufferWidth            = width;
    pp.BackBufferHeight           = height;
    pp.BackBufferFormat           = D3DFMT_UNKNOWN;
    pp.BackBufferCount            = 1;
    pp.MultiSampleType            = D3DMULTISAMPLE_NONE;
    pp.SwapEffect                 = D3DSWAPEFFECT_DISCARD;
    pp.hDeviceWindow              = hWnd;
    pp.Windowed                   = TRUE;
    pp.EnableAutoDepthStencil     = FALSE;
    pp.AutoDepthStencilFormat     = D3DFMT_UNKNOWN;
    pp.Flags                      = 0;
    pp.FullScreen_RefreshRateInHz = 0;
    pp.PresentationInterval       = D3DPRESENT_INTERVAL_ONE;

    IDirect3DDevice9* pDevice = NULL;

    {
        IDirect3D9* const pD3D = Direct3DCreate9(D3D_SDK_VERSION);
        if (!pD3D)
        {
            throw "Failed to initialize the D3D interface";
        }

        g_Config.adapter = D3DADAPTER_DEFAULT;

        D3DDISPLAYMODE displayMode;
        ZeroMemory(&displayMode, sizeof(displayMode));

        pD3D->GetAdapterDisplayMode(g_Config.adapter, &displayMode);

        g_Config.adapterFormat = displayMode.Format;
        g_Config.devType = D3DDEVTYPE_HAL;

        D3DFORMAT const spriteTextureFormats[] =
        {
            D3DFMT_A8R8G8B8,
            D3DFMT_A4R4G4B4,
            D3DFMT_A1R5G5B5,
        };
        g_Config.spriteTextureFormat = FindDeviceFormat(pD3D, 0, D3DRTYPE_TEXTURE, spriteTextureFormats);

        D3DFORMAT const backgroundTextureFormats[] =
        {
            D3DFMT_A8R8G8B8,
            D3DFMT_R5G6B5,
            D3DFMT_A1R5G5B5,
        };
        g_Config.backgroundTextureFormat = FindDeviceFormat(pD3D, 0, D3DRTYPE_TEXTURE, backgroundTextureFormats);

        D3DFORMAT const targetFormats[] =
        {
            D3DFMT_X8R8G8B8,
            D3DFMT_R5G6B5,
            D3DFMT_A1R5G5B5,
        };
        g_Config.targetFormat = FindBackBufferFormat(pD3D, targetFormats);

        HRESULT hr = pD3D->CreateDevice(
            g_Config.adapter,
            g_Config.devType,
            hWnd,
            D3DCREATE_MIXED_VERTEXPROCESSING,
            &pp,
            &pDevice);

        if (FAILED(hr))
        {
            hr = pD3D->CreateDevice(
                g_Config.adapter,
                g_Config.devType,
                hWnd,
                D3DCREATE_SOFTWARE_VERTEXPROCESSING,
                &pp,
                &pDevice);
        }

        if (FAILED(hr))
        {
            throw "Failed to initialize the D3D device";
        }

        // We don't need this anymore.

        pD3D->Release();
    }

    // Initialization: Load the textures

    IDirect3DTexture9* const pBackgroundTexture     = NewTexture(pDevice, "Background.png"    , g_Config.backgroundTextureFormat);
    IDirect3DTexture9* const pRedJewelTexture       = NewTexture(pDevice, "RedJewel.png"      , g_Config.spriteTextureFormat);
    IDirect3DTexture9* const pBlueJewelTexture      = NewTexture(pDevice, "BlueJewel.png"     , g_Config.spriteTextureFormat);
    IDirect3DTexture9* const pYellowJewelTexture    = NewTexture(pDevice, "YellowJewel.png"   , g_Config.spriteTextureFormat);
    IDirect3DTexture9* const pGreenJewelTexture     = NewTexture(pDevice, "GreenJewel.png"    , g_Config.spriteTextureFormat);
    IDirect3DTexture9* const pPinkJewelTexture      = NewTexture(pDevice, "PinkJewel.png"     , g_Config.spriteTextureFormat);
    IDirect3DTexture9* const pCyanJewelTexture      = NewTexture(pDevice, "CyanJewel.png"     , g_Config.spriteTextureFormat);
    IDirect3DTexture9* const pJewelSelectionTexture = NewTexture(pDevice, "JewelSelection.png", g_Config.spriteTextureFormat);

    IDirect3DTexture9* const jewelTextures[6] =
    {
        pRedJewelTexture,
        pBlueJewelTexture,
        pYellowJewelTexture,
        pGreenJewelTexture,
        pPinkJewelTexture,
        pCyanJewelTexture,
    };

    Board board;
    InitializeBoard(board, jewelTextures);

    FILETIME lastTime;
    GetSystemTimeAsFileTime(&lastTime);

    float swappingProgress = 0;
    float const swappingLength = 0.3f;

	// Main message loop:

    MSG msg = { 0 };

    while (msg.message != WM_QUIT)
    {
        // Measure time and update animations accordinly.
        // GetSystemTimeAsFileTime is a funny-sounding function,
        // but it offers better granularity than getTickCount.

        FILETIME currentTime;
        GetSystemTimeAsFileTime(&currentTime);

        // Filetime is in 100s of nanoseconds, so divide to get time in seconds.

        float const timeDelta = float(currentTime.dwLowDateTime - lastTime.dwLowDateTime) / 10000000.0f;
        lastTime = currentTime;

        if (g_State == SwappingJewels || g_State == SwappingJewelsBack)
        {
            swappingProgress += timeDelta;
            if (swappingProgress >= swappingLength)
            {
                swappingProgress = 0;

                if (g_State == SwappingJewels)
                {
                    SwapJewels(board, g_FirstSelectedX, g_FirstSelectedY, g_SecondSelectedX, g_SecondSelectedY);
                    if (ResolveMatches(board, jewelTextures))
                    {
                        g_State = Animating;
                    }
                    else
                    {
                        g_State = SwappingJewelsBack;
                    }
                }
                else
                {
                    SwapJewels(board, g_FirstSelectedX, g_FirstSelectedY, g_SecondSelectedX, g_SecondSelectedY);
                    g_State = NoneSelected;
                }
            }
            else
            {
                AnimateJewelSwap(swappingProgress / swappingLength, board, g_FirstSelectedX, g_FirstSelectedY, g_SecondSelectedX, g_SecondSelectedY);
            }
        }
        
        if (g_State == Animating)
        {
            // We'll stop the animation and enable input when nothing moves.

            if (!AnimateFallingJewels(timeDelta, board))
            {
                // The animation finished, so let's see if any matches were made.
                if (!ResolveMatches(board, jewelTextures))
                {
                    g_State = NoneSelected;
                }
            }
        }

        // Render.

        {
            pDevice->BeginScene();

            DrawTexturedOpaqueRectangle(pDevice, 0, 0, 800, 480, pBackgroundTexture);

            for (int x = 0; x < BoardWidth; ++x)
            {
                for (int y = 0; y < BoardHeight; ++y)
                {
                    Jewel& jewel = board.jewels[x][y];

                    float posx = jewel.x * 48 + 320;
                    float posy = jewel.y * 48;

                    DrawSpriteRectangle(pDevice, posx, posy, posx + 48, posy + 48, jewel.pTexture);
                }
            }

            static float angle = 0;
            angle += 0.1f * timeDelta;

            if (g_State == FirstSelected)
            {
                float const posx = float(g_FirstSelectedX) * 48 - 8 + 320;
                float const posy = float(g_FirstSelectedY) * 48 - 8;
                DrawRotatedSpriteRectangle(pDevice, posx, posy, posx + 64, posy + 64, angle, pJewelSelectionTexture);
            }

            pDevice->EndScene();

            pDevice->Present(NULL, NULL, NULL, NULL);
        }

        // Pump windows messages.

        while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) && msg.message != WM_QUIT)
	    {
		    TranslateMessage(&msg);
		    DispatchMessage(&msg);
	    }
    }

    pBackgroundTexture    ->Release();
    pRedJewelTexture      ->Release();
    pBlueJewelTexture     ->Release();
    pYellowJewelTexture   ->Release();
    pGreenJewelTexture    ->Release();
    pPinkJewelTexture     ->Release();
    pCyanJewelTexture     ->Release();
    pJewelSelectionTexture->Release();

    pDevice->Release();

	return (int) msg.wParam;
}

int APIENTRY wWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nCmdShow)
{
    int exitCode;

    try
    {
        exitCode = MyWinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow);
    }
    catch (char const* msg)
    {
        OutputDebugStringA(msg);
        MessageBoxA(NULL, msg, "Detected error", MB_OK);
        exitCode = 1;
    }

    return exitCode;
}
