#include "JewelBoard.h"

#include <Windows.h>
#include <assert.h>

#include <algorithm>
#include <Vector>

static
void GenerateNewJewels(Board& board, IDirect3DTexture9* const* jewelTextures, size_t numJewelTextures, bool avoidNewMatches)
{
    for (size_t x = 0; x < BoardWidth; ++x)
    {
        for (size_t y = 0; y < BoardHeight; ++y)
        {
            Jewel& jewel = board.jewels[x][y];

            if (jewel.pTexture == NULL)
            {
                // Initialize their position so they fall from the top

                jewel.x = float(x);
                jewel.y = -1.0f;
                jewel.fallVelocity = 0.0f;
                jewel.pTexture = jewelTextures[rand() % numJewelTextures];

                if (avoidNewMatches)
                {
                    HorizontalMatch hMatch = GetHorizontalMatch(board, x, y);
                    VerticalMatch vMatch = GetVerticalMatch(board, x, y);

                    // Ensure we don't generate new matches.

                    while (hMatch.w >= 3 || vMatch.h >= 3)
                    {
                        jewel.pTexture = jewelTextures[rand() % numJewelTextures];

                        hMatch = GetHorizontalMatch(board, x, y);
                        vMatch = GetVerticalMatch(board, x, y);
                    }
                }
            }
        }
    }
}

void InitializeBoard(Board& board, IDirect3DTexture9* const* jewelTextures, size_t numJewelTextures)
{
    ZeroMemory(&board, sizeof(board));

    GenerateNewJewels(board, jewelTextures, numJewelTextures, true);
}

void SwapJewels(Board& board, size_t x0, size_t y0, size_t x1, size_t y1)
{
    assert(x0 < BoardWidth);
    assert(y0 < BoardHeight);
    assert(x1 < BoardWidth);
    assert(y1 < BoardHeight);

    Jewel& jewel0 = board.jewels[x0][y0];
    Jewel& jewel1 = board.jewels[x1][y1];

    std::swap(jewel0.pTexture, jewel1.pTexture);

    jewel0.x = float(x0);
    jewel0.y = float(y0);
    jewel1.x = float(x1);
    jewel1.y = float(y1);
}

bool ResolveMatches(Board& board, IDirect3DTexture9* const* jewelTextures, size_t numJewelTextures)
{
    bool result = false;

    Board saveBoard = board;

    // First, clear out any jewels that were matched.

    std::vector<HorizontalMatch> horizontalMatches;
    std::vector<VerticalMatch> verticalMatches;

    for (size_t x = 0; x < BoardWidth; ++x)
    {
        for (size_t y = 0; y < BoardHeight; ++y)
        {
            Jewel& jewel = board.jewels[x][y];

            HorizontalMatch hMatch = GetHorizontalMatch(board, x, y);
            VerticalMatch vMatch = GetVerticalMatch(board, x, y);

            if (hMatch.w >= 3)
            {
                horizontalMatches.push_back(hMatch);
                result = true;
            }

            if (vMatch.h >= 3)
            {
                verticalMatches.push_back(vMatch);
                result = true;
            }
        }
    }

    if (result)
    {
        // Clear the matched jewels.

        auto const hEnd = horizontalMatches.end();
        auto const vEnd = verticalMatches.end();

        for (auto m = horizontalMatches.begin(); m != hEnd; ++m)
        {
            for (size_t i = 0; i < m->w; ++i)
            {
                board.jewels[i + m->x][m->y].pTexture = NULL;
            }
        }

        for (auto m = verticalMatches.begin(); m != vEnd; ++m)
        {
            for (size_t i = 0; i < m->h; ++i)
            {
                board.jewels[m->x][i + m->y].pTexture = NULL;
            }
        }

        // Now, "drop" the remaining jewels

        for (size_t x = 0; x < BoardWidth; ++x)
        {
            size_t dsty = BoardHeight;
            size_t srcy = BoardHeight;
            while (srcy > 0)
            {
                --srcy;

                Jewel const& srcjewel = board.jewels[x][srcy];

                if (srcjewel.pTexture != NULL)
                {
                    --dsty;
                    if (srcy < dsty)
                    {
                        board.jewels[x][dsty] = srcjewel;
                    }
                }
            }

            while (dsty > 0)
            {
                --dsty;
                board.jewels[x][dsty].pTexture = NULL;
            }
        }

        // And finally, generate new jewels

        GenerateNewJewels(board, jewelTextures, numJewelTextures, false);
    }

    for (size_t x = 0; x < BoardWidth; ++x)
    {
        for (size_t y = 0; y < BoardHeight; ++y)
        {
            Jewel& jewel = board.jewels[x][y];
            if (jewel.pTexture == NULL)
            {
                __debugbreak();
            }
        }
    }

    return result;
}

HorizontalMatch GetHorizontalMatch(Board const& board, size_t x, size_t y)
{
    assert(x < BoardWidth);
    assert(y < BoardHeight);

    HorizontalMatch result;

    IDirect3DTexture9* const pTexture = board.jewels[x][y].pTexture;

    size_t w = 1;
    while (x + w < BoardWidth && board.jewels[x + w][y].pTexture == pTexture)
    {
        ++w;
    }
    while (x > 0 && board.jewels[x - 1][y].pTexture == pTexture)
    {
        --x;
        ++w;
    }

    result.x = x;
    result.y = y;
    result.w = w;
    result.pTexture = pTexture;

    return result;
}

VerticalMatch GetVerticalMatch(Board const& board, size_t x, size_t y)
{
    assert(x < BoardWidth);
    assert(y < BoardHeight);

    VerticalMatch result;

    IDirect3DTexture9* const pTexture = board.jewels[x][y].pTexture;

    size_t h = 1;
    while (y + h < BoardHeight && board.jewels[x][y + h].pTexture == pTexture)
    {
        ++h;
    }
    while (y > 0 && board.jewels[x][y - 1].pTexture == pTexture)
    {
        --y;
        ++h;
    }

    result.x = x;
    result.y = y;
    result.h = h;
    result.pTexture = pTexture;

    return result;
}

bool AnimateFallingJewels(float timeDelta, Board& board)
{
    bool jewelsMoved = false;

    for (int x = 0; x < BoardWidth; ++x)
    {
        for (int y = 0; y < BoardHeight; ++y)
        {
            Jewel& jewel = board.jewels[x][y];

            if (jewel.y < float(y) && (y == BoardHeight - 1 || jewel.y /*+ 1 <=*/ < board.jewels[x][y+1].y))
            {
                jewelsMoved = true;

                jewel.fallVelocity += timeDelta;
                jewel.y += jewel.fallVelocity;

                if (jewel.y > float(y))
                {
                    jewel.y = float(y);
                }
            }
            else
            {
                jewel.fallVelocity = 0.0f;
            }
        }
    }

    return jewelsMoved;
}

void AnimateJewelSwap(float time, Board& board, size_t x0, size_t y0, size_t x1, size_t y1)
{
    Jewel& jewel0 = board.jewels[x0][y0];
    Jewel& jewel1 = board.jewels[x1][y1];

    jewel0.x = float(x0) + time * (float(x1) - float(x0));
    jewel0.y = float(y0) + time * (float(y1) - float(y0));
    jewel1.x = float(x1) - time * (float(x1) - float(x0));
    jewel1.y = float(y1) - time * (float(y1) - float(y0));
}
