#pragma once

#include <Windows.h>
#include <stddef.h>
#include <d3d9.h>
#include <d3dx9.h>

namespace HWRenderer
{

void Init(IDirect3DDevice9* pDevice);

// Pipeline structures
// These structures are used for two purposes:
// - to provide data to the renderer by the app.
// - to hold intermediate data inside of the pipeline.

struct VertexData
{
    // This structure contains all that "extra" data attached to a vertex or pixel.
    // This data gets copied, transformed or interpolated as appropriate in the pipeline,
    // and is ultimately consumed by the pixel shading function.

    D3DCOLOR    diffuse;
    D3DCOLOR    specular;
    D3DXVECTOR2 texCoord;
};

// The vertex data can get interpolated, so we define convenient arithmetic operators.

inline
VertexData operator+(VertexData const& data0, VertexData const& data1)
{
    VertexData result;

    result.diffuse  = D3DXCOLOR(data0.diffuse ) + D3DXCOLOR(data1.diffuse );
    result.specular = D3DXCOLOR(data0.specular) + D3DXCOLOR(data1.specular);
    result.texCoord = data0.texCoord + data1.texCoord;

    return result;
}

inline
VertexData operator*(VertexData const& data, float v)
{
    VertexData result;

    result.diffuse  = D3DXCOLOR(data.diffuse ) * v;
    result.specular = D3DXCOLOR(data.specular) * v;
    result.texCoord = data.texCoord * v;

    return result;
}

struct WorldVertex
{
    // This structure contains a vertex in world (or model) coordinates,
    // prior to calculating the light observed on the vertex.

    D3DXVECTOR3 position;
    D3DXVECTOR3 normal;
    D3DXVECTOR3 tangent;
    VertexData  data;

    static IDirect3DVertexDeclaration9* pVD;
    static D3DVERTEXELEMENT9 const declaration[];
};

// We define the vertex declaration here, close to its C++ structure.
// We must wrap it in #if HWRenderer_cpp so that the linker won't find multiple definitions.

#if HWRenderer_cpp
IDirect3DVertexDeclaration9* WorldVertex::pVD = NULL;
D3DVERTEXELEMENT9 const WorldVertex::declaration[] =
{
    { 0, offsetof(WorldVertex, position     ), D3DDECLTYPE_FLOAT3  , 0, D3DDECLUSAGE_POSITION, 0 },
    { 0, offsetof(WorldVertex, normal       ), D3DDECLTYPE_FLOAT3  , 0, D3DDECLUSAGE_NORMAL  , 0 },
    { 0, offsetof(WorldVertex, tangent      ), D3DDECLTYPE_FLOAT3  , 0, D3DDECLUSAGE_TANGENT , 0 },
    { 0, offsetof(WorldVertex, data.diffuse ), D3DDECLTYPE_D3DCOLOR, 0, D3DDECLUSAGE_COLOR   , 0 },
    { 0, offsetof(WorldVertex, data.specular), D3DDECLTYPE_D3DCOLOR, 0, D3DDECLUSAGE_COLOR   , 1 },
    { 0, offsetof(WorldVertex, data.texCoord), D3DDECLTYPE_FLOAT2  , 0, D3DDECLUSAGE_TEXCOORD, 0 },
    D3DDECL_END(),
};
#endif

struct LightedVertex
{
    // This structure contains a vertex after lights and projection
    // have been applied to it. It could be in clip or screen space.

    D3DXVECTOR4 position;
    VertexData  data;
    float       fog;

    static IDirect3DVertexDeclaration9* pVD;
    static D3DVERTEXELEMENT9 const declaration[];
};

// We define the vertex declaration here, close to its C++ structure.

#if HWRenderer_cpp
IDirect3DVertexDeclaration9* LightedVertex::pVD = NULL;
D3DVERTEXELEMENT9 const LightedVertex::declaration[] =
{
    { 0, offsetof(LightedVertex, position     ), D3DDECLTYPE_FLOAT4  , 0, D3DDECLUSAGE_POSITIONT, 0 },
    { 0, offsetof(LightedVertex, data.diffuse ), D3DDECLTYPE_D3DCOLOR, 0, D3DDECLUSAGE_COLOR    , 0 },
    { 0, offsetof(LightedVertex, data.specular), D3DDECLTYPE_D3DCOLOR, 0, D3DDECLUSAGE_COLOR    , 1 },
    { 0, offsetof(LightedVertex, data.texCoord), D3DDECLTYPE_FLOAT2  , 0, D3DDECLUSAGE_TEXCOORD , 0 },
    { 0, offsetof(LightedVertex, fog          ), D3DDECLTYPE_FLOAT1  , 0, D3DDECLUSAGE_FOG      , 0 },
    D3DDECL_END(),
};
#endif

// Rendering pipeline functions

struct VertexShadeParams
{
    D3DXVECTOR3 cameraPosition;
    D3DXMATRIX  lightingPosMatrix;
    D3DXMATRIX  lightingNormalMatrix;
    D3DXMATRIX  worldViewProjectionMatrix;

    D3DXCOLOR   ambientLightColor;

    D3DXCOLOR   directionalLightColor;
    D3DXVECTOR3 directionalLightDirection;

    D3DXCOLOR   pointLightColor;
    D3DXVECTOR3 pointLightPosition;
    float       pointLightRange;
    float       pointLightA0;
    float       pointLightA1;
    float       pointLightA2;

    float       specularPower;

    float       fogNear;
    float       fogFar;
};

} // namespace HWRenderer
