#define D3D_DEBUG_INFO
#define DIRECTINPUT_VERSION 0x0800

#include <Windows.h>
#include <string>
#include <d3d9.h>
#include <d3dx9.h>
#include <dinput.h>

#pragma comment(lib, "d3d9")
#pragma comment(lib, "d3dx9")
#pragma comment(lib, "dinput8")
#pragma comment(lib, "winmm")
#pragma comment(lib, "dxguid")

#include "HWRenderer.h"
#include "Utilities.h"

#include "VertexShade.vfxobj"
#include "LinearToGamma.pfxobj"
#include "PixelShade.pfxobj"
#include "PixelShadeTexture.pfxobj"


// ----------------------------------------

template < typename T = DWORD, unsigned int N = 32 >
class Averager
{
public:
    Averager()
        : pos(0)
        , accum(0)
    {
        memset(buf, 0, sizeof(buf));
    }

    void AddSample(T val)
    {
        accum -= buf[pos];
        accum += val;
        buf[pos] = val;
        pos = (pos + 1) % N;
    }

    double GetAverage() const
    {
        return double(accum) / N;
    }

private:
    T buf[N];
    T pos;
    T accum;
};

bool FileExists(char const* fileName)
{
    return GetFileAttributesA(fileName) != INVALID_FILE_ATTRIBUTES;
}

bool g_UseReferenceShaders    = false;

IDirect3DVertexShader9* g_pVFX_VertexShade       = NULL;
IDirect3DPixelShader9*  g_pPFX_LinearToGamma     = NULL;
IDirect3DPixelShader9*  g_pPFX_PixelShade        = NULL;
IDirect3DPixelShader9*  g_pPFX_PixelShadeTexture = NULL;

IDirect3DVertexShader9* g_pVFX_ReferenceVertexShade       = NULL;
IDirect3DPixelShader9*  g_pPFX_ReferencePixelShade        = NULL;
IDirect3DPixelShader9*  g_pPFX_ReferencePixelShadeTexture = NULL;

ID3DXBuffer* CompileShader(char const* fileName, char const* profile)
{
    if (!FileExists(fileName))
    {
        ::MessageBoxA(NULL, (std::string("Missing shader file (") + fileName + ")\nUsing the refrence implementation").c_str(), "Missing file", MB_OK);
        return NULL;
    }

    ID3DXBuffer* pDest = NULL;
    ID3DXBuffer* pErrorMsgs = NULL;

    HRESULT hr = D3DXCompileShaderFromFileA(fileName, NULL, NULL, "main", profile, D3DXSHADER_SKIPOPTIMIZATION | D3DXSHADER_DEBUG, &pDest, &pErrorMsgs, NULL);

    if (pErrorMsgs)
    {
        std::string errors;
        if (pErrorMsgs)
        {
            char const* start = (char const*) pErrorMsgs->GetBufferPointer();
            errors = std::string(start, start + pErrorMsgs->GetBufferSize());
        }

        if (pDest)
        {
            ::MessageBoxA(NULL, errors.c_str(), (std::string("Shader compilation warnings (") + fileName + ")").c_str(), MB_OK);
        }
        else
        {
            ::MessageBoxA(NULL, errors.c_str(), (std::string("Shader compilation errors (") + fileName + ")").c_str(), MB_OK);
        }

        pErrorMsgs->Release();
    }

    return pDest;
}

bool InitShaders(IDirect3DDevice9* pDevice)
{
    auto pVertexShade       = (DWORD const*)VFX_VertexShade;
    auto pLinearToGamma     = (DWORD const*)PFX_LinearToGamma;
    auto pPixelShade        = (DWORD const*)PFX_PixelShade;
    auto pPixelShadeTexture = (DWORD const*)PFX_PixelShadeTexture;

    ID3DXBuffer* pVertexShadeBuffer = CompileShader("VertexShade.vfx", "vs_2_0");
    if (pVertexShadeBuffer)
    {
        pVertexShade = (DWORD const*) pVertexShadeBuffer->GetBufferPointer();
    }

    //ID3DXBuffer* pLinearToGammaBuffer = CompileShader("LinearToGamma.pfx", "ps_2_0");
    //if (pLinearToGammaBuffer)
    //{
    //    pLinearToGamma = (DWORD const*) pLinearToGammaBuffer->GetBufferPointer();
    //}

    ID3DXBuffer* pPixelShadeBuffer = CompileShader("PixelShade.pfx", "ps_2_0");
    if (pPixelShadeBuffer)
    {
        pPixelShade = (DWORD const*) pPixelShadeBuffer->GetBufferPointer();
    }

    ID3DXBuffer* pPixelShadeTextureBuffer = CompileShader("PixelShadeTexture.pfx", "ps_2_0");
    if (pPixelShadeTextureBuffer)
    {
        pPixelShadeTexture = (DWORD const*) pPixelShadeTextureBuffer->GetBufferPointer();
    }

    bool result = SUCCEEDED(pDevice->CreateVertexShader(pVertexShade      , &g_pVFX_VertexShade      ))
              &&  SUCCEEDED(pDevice->CreatePixelShader (pLinearToGamma    , &g_pPFX_LinearToGamma    ))
              &&  SUCCEEDED(pDevice->CreatePixelShader (pPixelShade       , &g_pPFX_PixelShade       ))
              &&  SUCCEEDED(pDevice->CreatePixelShader (pPixelShadeTexture, &g_pPFX_PixelShadeTexture))
              &&  SUCCEEDED(pDevice->CreateVertexShader((DWORD const*)VFX_VertexShade      , &g_pVFX_ReferenceVertexShade      ))
              &&  SUCCEEDED(pDevice->CreatePixelShader ((DWORD const*)PFX_PixelShade       , &g_pPFX_ReferencePixelShade       ))
              &&  SUCCEEDED(pDevice->CreatePixelShader ((DWORD const*)PFX_PixelShadeTexture, &g_pPFX_ReferencePixelShadeTexture))
              &&  true;

    if (pVertexShadeBuffer)       pVertexShadeBuffer      ->Release();
    //if (pLinearToGammaBuffer)     pLinearToGammaBuffer    ->Release();
    if (pPixelShadeBuffer)        pPixelShadeBuffer       ->Release();
    if (pPixelShadeTextureBuffer) pPixelShadeTextureBuffer->Release();

    return result;
}

typedef float (*FuncT)(float);

float Gamma22(float x)
{
    return pow(x, 2.2f);
}

float Gamma22Inv(float x)
{
    return pow(x, 1/2.2f);
}

float Gamma22Approx(float x)
{
    float const x2 = x * x;

    return x2 * 0.82740075f + x2 * x2 * 0.17259925f;
}

float Gamma22InvApprox(float x)
{
    static float const B = 8 * pow(0.5f, 1/2.2f) - 4;
    static float const C = -2 * B;
    static float const D = 1 + B;

    return D * x + C * x*x + B * x*x*x;
/*
    float const x2 = sqrt(x);

    return x2 * 0.82740075f + sqrt(x2) * 0.17259925f;
*/
}

template < class FuncT >
void RenderFunc(IDirect3DDevice9* pDevice, FuncT const& func, D3DXCOLOR const& c0)
{
    D3DSURFACE_DESC targetDesc;
    {
        IDirect3DSurface9* pTarget = NULL;
        pDevice->GetRenderTarget(0, &pTarget);
        pTarget->GetDesc(&targetDesc);
        pTarget->Release();
    }
    float const width  = float(targetDesc.Width);
    float const height = float(targetDesc.Height);

    unsigned int const NumVertices = 64;
    HWRenderer::LightedVertex vertices[NumVertices];
    
    for (DWORD i = 0; i < NumVertices; ++i)
    {
        float const x = float(i) / float(NumVertices-1);
        float const y = func(x);

        vertices[i].position      = D3DXVECTOR4(width * x, height * y, 1.0f, 1.0f);
        vertices[i].data.diffuse  = c0;
        vertices[i].data.specular = c0;
        vertices[i].data.texCoord = D3DXVECTOR2(0, 0);
        vertices[i].fog           = 0;
    }

    pDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_FALSE);
    pDevice->SetRenderState(D3DRS_ZWRITEENABLE, FALSE);

    pDevice->SetPixelShader(g_pPFX_PixelShade);
    pDevice->SetTexture(0, NULL);
    pDevice->SetVertexDeclaration(HWRenderer::LightedVertex::pVD);

    pDevice->DrawPrimitiveUP(D3DPT_LINESTRIP, NumVertices-1, vertices, sizeof(vertices[0]));
}

void RenderBackground(
    IDirect3DDevice9* pDevice, IDirect3DTexture9* pTexture,
    D3DXVECTOR2 const& t0 = D3DXVECTOR2(0, 0), D3DXVECTOR2 const& t1 = D3DXVECTOR2(1, 1),
    bool colored = true)
{
    D3DSURFACE_DESC targetDesc;
    {
        IDirect3DSurface9* pTarget = NULL;
        pDevice->GetRenderTarget(0, &pTarget);
        pTarget->GetDesc(&targetDesc);
        pTarget->Release();
    }
    float const width  = float(targetDesc.Width);
    float const height = float(targetDesc.Height);

    D3DXCOLOR const c0 = colored? D3DXCOLOR(1, 0, 1, 1): D3DXWhite;
    D3DXCOLOR const c1 = colored? D3DXCOLOR(1, 0, 0, 1): D3DXWhite;
    D3DXCOLOR const c2 = colored? D3DXCOLOR(0, 1, 1, 1): D3DXWhite;
    D3DXCOLOR const c3 = colored? D3DXCOLOR(0, 1, 0, 1): D3DXWhite;

    unsigned int const NumVertices = 4;
    HWRenderer::LightedVertex const vertices[NumVertices] =
    {
        { D3DXVECTOR4(    0, height, 1.0f, 1.0f), { c0, D3DXBlack, D3DXVECTOR2(t0.x, t1.y) }, 0 },
        { D3DXVECTOR4(    0,      0, 1.0f, 1.0f), { c1, D3DXBlack, D3DXVECTOR2(t0.x, t0.y) }, 0 },
        { D3DXVECTOR4(width, height, 1.0f, 1.0f), { c2, D3DXBlack, D3DXVECTOR2(t1.x, t1.y) }, 0 },
        { D3DXVECTOR4(width,      0, 1.0f, 1.0f), { c3, D3DXBlack, D3DXVECTOR2(t1.x, t0.y) }, 0 },
    };

    pDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_FALSE);
    pDevice->SetRenderState(D3DRS_ZWRITEENABLE, FALSE);

    pDevice->SetPixelShader(pTexture? g_UseReferenceShaders? g_pPFX_ReferencePixelShadeTexture: g_pPFX_PixelShadeTexture: g_UseReferenceShaders? g_pPFX_ReferencePixelShade: g_pPFX_PixelShade);
    pDevice->SetTexture(0, pTexture);
    pDevice->SetVertexDeclaration(HWRenderer::LightedVertex::pVD);

    pDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, vertices, sizeof(vertices[0]));
}

void RenderCube(IDirect3DDevice9* pDevice)
{
    static float const normalScale = 1.0f / sqrt(3.0f);

    static DWORD const NumVertices = 8;
    static HWRenderer::WorldVertex const vertices[NumVertices] =
    {
        { D3DXVECTOR3(-1.0f, -1.0f, -1.0f), normalScale * D3DXVECTOR3(-1.0f, -1.0f, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(0, 0, 0, 1), D3DXWhite, D3DXVECTOR2(0, 0) } },
        { D3DXVECTOR3( 1.0f, -1.0f, -1.0f), normalScale * D3DXVECTOR3( 1.0f, -1.0f, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(1, 0, 0, 1), D3DXWhite, D3DXVECTOR2(1, 0) } },
        { D3DXVECTOR3( 1.0f,  1.0f, -1.0f), normalScale * D3DXVECTOR3( 1.0f,  1.0f, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(1, 1, 0, 1), D3DXWhite, D3DXVECTOR2(1, 1) } },
        { D3DXVECTOR3(-1.0f,  1.0f, -1.0f), normalScale * D3DXVECTOR3(-1.0f,  1.0f, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(0, 1, 0, 1), D3DXWhite, D3DXVECTOR2(0, 1) } },

        { D3DXVECTOR3(-1.0f, -1.0f,  1.0f), normalScale * D3DXVECTOR3(-1.0f, -1.0f,  1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(0, 0, 1, 1), D3DXWhite, D3DXVECTOR2(0, 0) } },
        { D3DXVECTOR3( 1.0f, -1.0f,  1.0f), normalScale * D3DXVECTOR3( 1.0f, -1.0f,  1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(1, 0, 1, 1), D3DXWhite, D3DXVECTOR2(1, 0) } },
        { D3DXVECTOR3( 1.0f,  1.0f,  1.0f), normalScale * D3DXVECTOR3( 1.0f,  1.0f,  1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(1, 1, 1, 1), D3DXWhite, D3DXVECTOR2(1, 1) } },
        { D3DXVECTOR3(-1.0f,  1.0f,  1.0f), normalScale * D3DXVECTOR3(-1.0f,  1.0f,  1.0f), D3DXVECTOR3(0, 0, 0), { D3DXCOLOR(0, 1, 1, 1), D3DXWhite, D3DXVECTOR2(0, 1) } },
    };

    static DWORD const NumTriangles = 12;
    static WORD const indices[NumTriangles * 3] =
    {
        0, 2, 1,  0, 3, 2,
        4, 5, 6,  4, 6, 7,
        1, 4, 0,  1, 5, 4,
        2, 5, 1,  2, 6, 5,
        3, 6, 2,  3, 7, 6,
        0, 7, 3,  0, 4, 7,
    };

    pDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_TRUE);
    pDevice->SetRenderState(D3DRS_ZWRITEENABLE, TRUE);

    pDevice->SetVertexDeclaration(HWRenderer::WorldVertex::pVD);

    pDevice->DrawIndexedPrimitiveUP(D3DPT_TRIANGLELIST, 0, NumVertices, NumTriangles, indices, D3DFMT_INDEX16, vertices, sizeof(vertices[0]));
}

void RenderBillboard(IDirect3DDevice9* pDevice)
{
    static float const normalScale = 1.0f / sqrt(3.0f);

    static DWORD const NumVertices = 4;
    static HWRenderer::WorldVertex const vertices[NumVertices] =
    {
        { D3DXVECTOR3(-1.0f,  1.0f, 0.0f), D3DXVECTOR3(0, 0, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXWhite, D3DXWhite, D3DXVECTOR2(0, 0) } },
        { D3DXVECTOR3( 1.0f,  1.0f, 0.0f), D3DXVECTOR3(0, 0, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXWhite, D3DXWhite, D3DXVECTOR2(1, 0) } },
        { D3DXVECTOR3( 1.0f, -1.0f, 0.0f), D3DXVECTOR3(0, 0, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXWhite, D3DXWhite, D3DXVECTOR2(1, 1) } },
        { D3DXVECTOR3(-1.0f, -1.0f, 0.0f), D3DXVECTOR3(0, 0, -1.0f), D3DXVECTOR3(0, 0, 0), { D3DXWhite, D3DXWhite, D3DXVECTOR2(0, 1) } },
    };

    static DWORD const NumTriangles = 2;
    static WORD const indices[NumTriangles * 3] =
    {
        0, 2, 1,  0, 3, 2,
    };

    pDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_TRUE);
    pDevice->SetRenderState(D3DRS_ZWRITEENABLE, TRUE);

    pDevice->SetVertexDeclaration(HWRenderer::WorldVertex::pVD);

    pDevice->DrawIndexedPrimitiveUP(D3DPT_TRIANGLELIST, 0, NumVertices, NumTriangles, indices, D3DFMT_INDEX16, vertices, sizeof(vertices[0]));
}

struct Mesh
{
    DWORD                   numVertices;
    DWORD                   numTriangles;
    IDirect3DVertexBuffer9* vertices;
    IDirect3DIndexBuffer9*  indices;
};

Mesh ConvertMesh(IDirect3DDevice9* pDevice, ID3DXMesh* pMesh, D3DXCOLOR const& diffuse = D3DXWhite, D3DXCOLOR const& specular = D3DXWhite)
{
    Mesh mesh;

    mesh.numVertices  = pMesh->GetNumVertices();
    mesh.numTriangles = pMesh->GetNumFaces();

    pDevice->CreateVertexBuffer(mesh.numVertices * sizeof(HWRenderer::WorldVertex), D3DUSAGE_WRITEONLY, 0, D3DPOOL_MANAGED, &mesh.vertices, NULL);
    pDevice->CreateIndexBuffer (mesh.numTriangles * 6, D3DUSAGE_WRITEONLY, D3DFMT_INDEX16, D3DPOOL_MANAGED, &mesh.indices, NULL);

    D3DXVECTOR3* vertices = NULL;
    WORD*        indices  = NULL;

    pMesh->LockVertexBuffer(D3DLOCK_READONLY, (LPVOID*)&vertices);
    pMesh->LockIndexBuffer (D3DLOCK_READONLY, (LPVOID*)&indices);

    HWRenderer::WorldVertex* meshVertices = NULL;
    WORD*                    meshIndices  = NULL;

    mesh.vertices->Lock(0, mesh.numVertices  * sizeof(meshVertices[0]), (void**)&meshVertices, 0);
    mesh.indices ->Lock(0, mesh.numTriangles * 6                      , (void**)&meshIndices , 0);

    for (DWORD i = 0; i < mesh.numVertices; ++i)
    {
        float const PI    = 3.141592f;
        float const DBLPI = PI * 2;

        D3DXVECTOR3 const& normal = vertices[i*2 + 1];

        meshVertices[i].position        = vertices[i*2];
        meshVertices[i].normal          = normal;
        meshVertices[i].data.diffuse    = diffuse;
        meshVertices[i].data.specular   = specular;
        meshVertices[i].data.texCoord.x = (atan2(normal.z, normal.x) + PI) / DBLPI;
        meshVertices[i].data.texCoord.y = acos(normal.y) / PI;

        D3DXVec3Cross(&meshVertices[i].tangent, &normal, &D3DXVECTOR3(0, 1, 0));
        D3DXVec3Normalize(&meshVertices[i].tangent, &meshVertices[i].tangent);
    }

    for (DWORD i = 0; i < mesh.numTriangles*3; ++i)
    {
        meshIndices[i] = indices[i];
    }

    mesh.indices ->Unlock();
    mesh.vertices->Unlock();

    pMesh->UnlockIndexBuffer ();
    pMesh->UnlockVertexBuffer();

    return mesh;
}

float GridFetch16(D3DLOCKED_RECT const& lr, DWORD x, DWORD y)
{
    UINT64 const value = ((UINT64 const*)(ULONG_PTR(lr.pBits) + lr.Pitch * y))[x];

    return (value & 0xFFFF) / float(0xFFFF);
}

float GridFetch10(D3DLOCKED_RECT const& lr, DWORD x, DWORD y)
{
    UINT32 const value = ((UINT32 const*)(ULONG_PTR(lr.pBits) + lr.Pitch * y))[x];

    return (value & 0x3FF) / float(0x3FF);
}

float GridFetch8(D3DLOCKED_RECT const& lr, DWORD x, DWORD y)
{
    UINT32 const value = ((UINT32 const*)(ULONG_PTR(lr.pBits) + lr.Pitch * y))[x];

    return (value & 0xFF) / float(0xFF);
}

struct GridFetch
{
    GridFetch() {}
    GridFetch(DWORD _w, DWORD _h, D3DFORMAT format, D3DLOCKED_RECT const& _lr): w(float(_w)), h(float(_h)), lr(_lr)
    {
        switch (format)
        {
        default:
        case D3DFMT_A8R8G8B8    : fetch = &GridFetch8 ; break;
        case D3DFMT_A2B10G10R10 : fetch = &GridFetch10; break;
        case D3DFMT_A2R10G10B10 : fetch = &GridFetch10; break;
        case D3DFMT_A16B16G16R16: fetch = &GridFetch16; break;
        }
    }

    float operator()(float x, float y) const
    {
        return fetch(lr, DWORD(floor(x * w)), DWORD(floor(y * h)));
    }

private:
    float          w;
    float          h;
    D3DLOCKED_RECT lr;

    float (*fetch)(D3DLOCKED_RECT const& lr, DWORD x, DWORD y);
};

Mesh CreateGridMesh(IDirect3DDevice9* pDevice, IDirect3DTexture9* pHeightMap, unsigned int tilesX, unsigned int tilesZ, float sizeX, float sizeY, float sizeZ, D3DXVECTOR3 const& center, D3DXCOLOR const& diffuse = D3DXWhite, D3DXCOLOR const& specular = D3DXWhite)
{
    Mesh mesh;

    mesh.numVertices  = (tilesX + 1) * (tilesZ + 1);
    mesh.numTriangles = tilesX * tilesZ * 2;

    pDevice->CreateVertexBuffer(mesh.numVertices * sizeof(HWRenderer::WorldVertex), D3DUSAGE_WRITEONLY, 0, D3DPOOL_MANAGED, &mesh.vertices, NULL);
    pDevice->CreateIndexBuffer (mesh.numTriangles * 6, D3DUSAGE_WRITEONLY, D3DFMT_INDEX16, D3DPOOL_MANAGED, &mesh.indices, NULL);

    D3DXVECTOR3 const normal(0, 1, 0);
    D3DXVECTOR3 const vX(sizeX, 0, 0);
    D3DXVECTOR3 const vY(0, sizeY, 0);
    D3DXVECTOR3 const vZ(0, 0, sizeZ);

    float const grad = 0.5f / float(tilesX + tilesZ);

    DWORD mipmap = 0;

    GridFetch gridFetch;

    if (pHeightMap)
    {
        D3DSURFACE_DESC desc;
        pHeightMap->GetLevelDesc(0, &desc);
        DWORD const width  = desc.Width;
        DWORD const height = desc.Height;

        static float const loge2 = 2 * log(2.0f);

        float const loge = log(float(width * height) / float(tilesX * tilesZ));

        float const fmipmap = max(loge / loge2, 0);

        mipmap = DWORD(floor(fmipmap));

        D3DLOCKED_RECT lr;

        while (FAILED(pHeightMap->LockRect(mipmap, &lr, NULL, D3DLOCK_READONLY)))
        {
            --mipmap;
        }

        pHeightMap->GetLevelDesc(mipmap, &desc);
        DWORD const mwidth  = desc.Width;
        DWORD const mheight = desc.Height;

        gridFetch = GridFetch(mwidth, mheight, desc.Format, lr);
    }

    D3DXVECTOR3 const start = center - D3DXVECTOR3(sizeX * tilesX / 2, 0, sizeZ * tilesZ / 2);

    HWRenderer::WorldVertex* meshVertices = NULL;
    WORD*                    meshIndices  = NULL;

    mesh.vertices->Lock(0, mesh.numVertices  * sizeof(meshVertices[0]), (void**)&meshVertices, 0);
    mesh.indices ->Lock(0, mesh.numTriangles * 6                      , (void**)&meshIndices , 0);

    for (unsigned int z = 0; z <= tilesZ; ++z)
    {
        D3DXVECTOR3 const startZ = start + vZ * float(z);

        for (unsigned int x = 0; x <= tilesX; ++x)
        {
            unsigned int const i = z * (tilesX + 1) + x;

            float const tx = float(x) / float(tilesX);
            float const ty = float(z) / float(tilesZ);

            D3DXVECTOR3 const pos = startZ + vX * float(x);

            if (pHeightMap)
            {
                float const heightAdjust = gridFetch(tx, ty);
                float const gradNx       = gridFetch(tx-grad, ty);
                float const gradPx       = gridFetch(tx+grad, ty);
                float const gradNz       = gridFetch(tx, ty-grad);
                float const gradPz       = gridFetch(tx, ty+grad);

                D3DXVECTOR3 const gradX(grad * 2 * tilesX * sizeX, (gradPx - gradNx) * sizeY, 0);
                D3DXVECTOR3 const gradZ(0, (gradPz - gradNz) * sizeY, grad * 2 * tilesZ * sizeZ);
                D3DXVECTOR3 up;

                D3DXVec3Cross(&up, &gradZ, &gradX);
                D3DXVec3Normalize(&meshVertices[i].normal, &up);

                meshVertices[i].position    = pos + vY * heightAdjust;
                meshVertices[i].tangent     = D3DXVECTOR3(0, 0, 0);
            }
            else
            {
                meshVertices[i].position    = pos;
                meshVertices[i].normal      = normal;
                meshVertices[i].tangent     = D3DXVECTOR3(0, 0, 0);
            }

            meshVertices[i].data.diffuse    = diffuse;
            meshVertices[i].data.specular   = specular;
            meshVertices[i].data.texCoord.x = tx;
            meshVertices[i].data.texCoord.y = ty;
        }
    }

    for (unsigned int z = 0; z < tilesZ; ++z)
    {
        unsigned int const indexZ = z * (tilesX + 1);

        for (unsigned int x = 0; x < tilesX; ++x)
        {
            unsigned int const i = (z * tilesX + x) * 6;

            meshIndices[i  ] = indexZ + x;
            meshIndices[i+2] = indexZ + x + tilesZ + 2;
            meshIndices[i+1] = indexZ + x + tilesZ + 1;
            meshIndices[i+3] = indexZ + x;
            meshIndices[i+4] = indexZ + x + tilesZ + 2;
            meshIndices[i+5] = indexZ + x + 1;
        }
    }

    if (pHeightMap)
    {
        pHeightMap->UnlockRect(mipmap);
    }

    mesh.indices ->Unlock();
    mesh.vertices->Unlock();

    return mesh;
}

void RenderMesh(IDirect3DDevice9* pDevice, Mesh const& mesh)
{

    pDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_TRUE);
    pDevice->SetRenderState(D3DRS_ZWRITEENABLE, TRUE);

    pDevice->SetVertexDeclaration(HWRenderer::WorldVertex::pVD);

    pDevice->SetStreamSource(0, mesh.vertices, 0, sizeof(HWRenderer::WorldVertex));
    pDevice->SetIndices     (mesh.indices);

    pDevice->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, 0, 0, mesh.numVertices, 0, mesh.numTriangles);
}

bool run = true;
bool reset = false;
int increment = 0;

bool g_UseGammaCorrection     = true;
unsigned int g_RenderWhich     = 0;
unsigned int g_RenderWhichNext = 0;
unsigned int g_RenderWhat     = 0;
unsigned int g_RenderWhatNext = 0;
bool g_FogEnable              = true;
bool g_DirectionalLightEnable = true;
bool g_PointLightEnable       = true;
bool g_SpecularLightEnable    = true;
bool g_UseGridTexture         = false;
bool g_UseAnisotropicLighting = false;

D3DXVECTOR3 g_CameraEye   = D3DXVECTOR3(0, 2.5f, -5);
float g_CameraYaw   = 0;
float g_CameraPitch = 0.5f;

DWORD                g_BackgroundAddressingIndex = 2;
DWORD                g_BackgroundFilterIndex     = 2;
D3DTEXTUREADDRESS    g_BackgroundAddressing = D3DTADDRESS_MIRROR;
D3DTEXTUREFILTERTYPE g_BackgroundFilter     = D3DTEXF_LINEAR;
D3DTEXTUREFILTERTYPE g_BackgroundMipFilter  = D3DTEXF_LINEAR;

D3DTEXTUREADDRESS const g_BackgroundAddressingList[3] =
{
    D3DTADDRESS_CLAMP,
    D3DTADDRESS_WRAP,
    D3DTADDRESS_MIRROR
};

D3DTEXTUREFILTERTYPE const g_BackgroundFilterList[3][2]  =
{
    { D3DTEXF_POINT , D3DTEXF_POINT  },
    { D3DTEXF_LINEAR, D3DTEXF_POINT  },
    { D3DTEXF_LINEAR, D3DTEXF_LINEAR },
};

float const g_Speed = 0.4f;

Mesh g_GroundMesh;
Mesh g_FloorMesh;

Mesh g_TeapotMesh;
Mesh g_SphereMesh;

void SetShaderTransforms(D3DXVECTOR4 (&VSConstants)[19], D3DXMATRIX const& world, D3DXMATRIX const& view, D3DXMATRIX const& proj)
{
    D3DXMATRIX m, m2;

    D3DXMatrixTranspose(&m, &(world * view * proj));
    *(D3DXMATRIX*)&VSConstants[0] = m;

    D3DXMatrixTranspose(&m, &world);
    *(D3DXVECTOR4*)&VSConstants[4] = D3DXVECTOR4(m.m[0]);
    *(D3DXVECTOR4*)&VSConstants[5] = D3DXVECTOR4(m.m[1]);
    *(D3DXVECTOR4*)&VSConstants[6] = D3DXVECTOR4(m.m[2]);

    D3DXMatrixInverse(&m2, NULL, &m);
    D3DXMatrixTranspose(&m, &m2);
    *(D3DXVECTOR4*)&VSConstants[7] = D3DXVECTOR4(m.m[0]);
    *(D3DXVECTOR4*)&VSConstants[8] = D3DXVECTOR4(m.m[1]);
    *(D3DXVECTOR4*)&VSConstants[9] = D3DXVECTOR4(m.m[2]);
}

D3DXCOLOR LinearToGamma(D3DXCOLOR const& c)
{
    return D3DXCOLOR(pow(c.r, 1/2.2f), pow(c.g, 1/2.2f), pow(c.b, 1/2.2f), c.a);
}

D3DXCOLOR GammaToLinear(D3DXCOLOR const& c)
{
    return D3DXCOLOR(pow(c.r, 2.2f), pow(c.g, 2.2f), pow(c.b, 2.2f), c.a);
}

std::string RenderScene(IDirect3DDevice9* pDevice, IDirect3DTexture9* pTexture, float currentTime)
{
    char const* whatName = "";

    D3DXCOLOR skyColor(0.5f, 1, 1, 1);

    if (!g_UseGammaCorrection)
    {
        skyColor = LinearToGamma(skyColor);
    }

    // Clear the render target buffers with sensible values.

    pDevice->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, skyColor, 1, 0);

    // Calculate the view/projection matrices.

    D3DXMATRIX view;
    D3DXMATRIX proj;

    {
        IDirect3DSurface9* pBack = NULL;;
        pDevice->GetBackBuffer(0, 0, D3DBACKBUFFER_TYPE_MONO, &pBack);

        D3DSURFACE_DESC desc;
        pBack->GetDesc(&desc);

        pBack->Release();

        D3DXMatrixPerspectiveFovLH(&proj, 1, float(desc.Width)/float(desc.Height), 0.01f, 100);
    }

    {
        D3DXMATRIX const iview = MatrixScaleYawPitchRollPos(D3DXVECTOR3(1, 1, 1), g_CameraYaw, g_CameraPitch, 0, g_CameraEye);

        D3DXMatrixInverse(&view, NULL, &iview);
    }

    pDevice->SetTransform(D3DTS_VIEW      , &view);
    pDevice->SetTransform(D3DTS_PROJECTION, &proj);

    // Set up lighting.

    D3DXVECTOR4 VSConstants[19];
    memset(&VSConstants, 0, sizeof(VSConstants));

    *(D3DXVECTOR3*)&VSConstants[10] = g_CameraEye;

    {
        float const light = 0.073f * currentTime;

        pDevice->SetRenderState(D3DRS_LIGHTING              , TRUE);
        pDevice->SetRenderState(D3DRS_SPECULARENABLE        , TRUE);
        pDevice->SetRenderState(D3DRS_NORMALIZENORMALS      , TRUE);
        pDevice->SetRenderState(D3DRS_LOCALVIEWER           , TRUE);
        pDevice->SetRenderState(D3DRS_COLORVERTEX           , TRUE);
        pDevice->SetRenderState(D3DRS_AMBIENTMATERIALSOURCE , D3DMCS_COLOR1);
        pDevice->SetRenderState(D3DRS_DIFFUSEMATERIALSOURCE , D3DMCS_COLOR1);
        pDevice->SetRenderState(D3DRS_SPECULARMATERIALSOURCE, D3DMCS_COLOR2);
        pDevice->SetRenderState(D3DRS_EMISSIVEMATERIALSOURCE, D3DMCS_MATERIAL);

        D3DXCOLOR ambientColor(0.05f, 0.05f, 0.05f, 0.5f);

        if (!g_UseGammaCorrection)
        {
            ambientColor = LinearToGamma(ambientColor);
        }

        *(D3DXCOLOR*)&VSConstants[11] = ambientColor;
        pDevice->SetRenderState(D3DRS_AMBIENT, D3DCOLOR(ambientColor));

        {
            D3DLIGHT9 d3dlight = { D3DLIGHT_DIRECTIONAL };

            d3dlight.Diffuse   =
            d3dlight.Specular  = D3DXCOLOR(0.8f, 0.6f, 0.4f, 1) * (g_DirectionalLightEnable? 1.0f: 0.0f);
            d3dlight.Direction = D3DXVECTOR3(-0.7f, -0.7f, 0);

            if (!g_UseGammaCorrection)
            {
                d3dlight.Diffuse  =
                d3dlight.Specular = LinearToGamma(d3dlight.Diffuse);
            }

            *(D3DXCOLOR*  )&VSConstants[12] = d3dlight.Diffuse;
            *(D3DXVECTOR3*)&VSConstants[13] = d3dlight.Direction;

            pDevice->SetLight(0, &d3dlight);
            pDevice->LightEnable(0, TRUE);
        }

        {
            D3DLIGHT9 d3dlight = { D3DLIGHT_POINT };

            d3dlight.Diffuse      = D3DXCOLOR(0, 1, 0, 1) * (g_PointLightEnable? 1.0f: 0.0f);
            d3dlight.Specular     = D3DXCOLOR(0, 1, 0, 1) * (g_PointLightEnable? 1.0f: 0.0f);
            d3dlight.Position     = D3DXVECTOR3(10 * cos(light), 0, 10 * sin(light));
            d3dlight.Range        = 20;
            d3dlight.Attenuation0 = 1;
            d3dlight.Attenuation1 = 0;
            d3dlight.Attenuation2 = 1.0f/50;

            if (!g_UseGammaCorrection)
            {
                d3dlight.Diffuse  =
                d3dlight.Specular = LinearToGamma(d3dlight.Diffuse);
            }

            *(D3DXCOLOR*  )&VSConstants[14] = d3dlight.Diffuse;
            *(D3DXVECTOR3*)&VSConstants[15] = d3dlight.Position;
            VSConstants[16].x = d3dlight.Attenuation0;
            VSConstants[16].y = d3dlight.Attenuation1;
            VSConstants[16].z = d3dlight.Attenuation2;
            VSConstants[16].w = d3dlight.Range;

            pDevice->SetLight(1, &d3dlight);
            pDevice->LightEnable(1, TRUE);
        }

        {
            D3DMATERIAL9 material = { 0 };

            material.Power = 8;

            VSConstants[17].x = material.Power;
            VSConstants[18].x = g_SpecularLightEnable? 1.0f: 0.0f;

            if (g_SpecularLightEnable)
            {
                material.Specular.r = 0.0f;
                material.Specular.g = 0.0f;
                material.Specular.b = 0.0f;

                pDevice->SetRenderState(D3DRS_SPECULARMATERIALSOURCE, D3DMCS_MATERIAL);
            }
            else
            {
                pDevice->SetRenderState(D3DRS_SPECULARMATERIALSOURCE, D3DMCS_COLOR2);
            }

            pDevice->SetMaterial(&material);
        }

        float const fogNear = g_FogEnable? 10:  99.9f;
        float const fogFar  = g_FogEnable? 40: 100.0f;

        VSConstants[17].y = fogNear;
        VSConstants[17].z = fogFar;
        VSConstants[17].w = g_UseAnisotropicLighting? 1.0f: 0.0f;

        pDevice->SetRenderState(D3DRS_FOGENABLE     , TRUE);
        pDevice->SetRenderState(D3DRS_FOGVERTEXMODE , D3DFOG_LINEAR);
        pDevice->SetRenderState(D3DRS_RANGEFOGENABLE, TRUE);
        pDevice->SetRenderState(D3DRS_FOGSTART      , FloatToRawDWORD(fogNear));
        pDevice->SetRenderState(D3DRS_FOGEND        , FloatToRawDWORD(fogFar));
        pDevice->SetRenderState(D3DRS_FOGCOLOR      , skyColor);
    }

    pDevice->SetVertexShader(g_UseReferenceShaders? g_pVFX_ReferenceVertexShade: g_pVFX_VertexShade);

    // Render the ground.

    {
        D3DXMATRIX world;

        D3DXMatrixIdentity(&world);

        pDevice->SetTransform(D3DTS_WORLD, &world);

        SetShaderTransforms(VSConstants, world, view, proj);
        pDevice->SetVertexShaderConstantF(0, (float const*)VSConstants, 19);
    }

    pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
    pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShade: g_pPFX_PixelShade);
    pDevice->SetTexture(0, NULL);

    RenderMesh(pDevice, g_GroundMesh);

    // Render the spinning object.

    {
        D3DXMATRIX world;

        float const yaw   = 0.01f  * currentTime;
        float const pitch = 0.037f * currentTime;
        float const roll  = 0.007f * currentTime;

        D3DXMatrixRotationYawPitchRoll(&world, yaw, pitch, roll);

        pDevice->SetTransform(D3DTS_WORLD, &world);

        SetShaderTransforms(VSConstants, world, view, proj);
        pDevice->SetVertexShaderConstantF(0, (float const*)VSConstants, 19);

        pDevice->SetRenderState(D3DRS_SRCBLEND , D3DBLEND_SRCALPHA);
        pDevice->SetRenderState(D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA);
        pDevice->SetRenderState(D3DRS_BLENDOP  , D3DBLENDOP_ADD);

        pDevice->SetSamplerState(0, D3DSAMP_ADDRESSU , D3DTADDRESS_CLAMP);
        pDevice->SetSamplerState(0, D3DSAMP_ADDRESSV , D3DTADDRESS_CLAMP);
        pDevice->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
        pDevice->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
        pDevice->SetSamplerState(0, D3DSAMP_MIPFILTER, D3DTEXF_LINEAR);


        switch (g_RenderWhat)
        {
            case 0:
                whatName = "Cube";
                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
                pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShadeTexture: g_pPFX_PixelShadeTexture);
                pDevice->SetTexture(0, pTexture);
                RenderCube(pDevice);
                g_RenderWhatNext = g_RenderWhat + 1;
                break;
            case 1:
                whatName = "Flat teapot";
                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
                pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShade: g_pPFX_PixelShade);
                pDevice->SetTexture(0, NULL);
                RenderMesh(pDevice, g_TeapotMesh);
                g_RenderWhatNext = g_RenderWhat + 1;
                break;
            case 2:
                whatName = "Teapot";
                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
                pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShadeTexture: g_pPFX_PixelShadeTexture);
                pDevice->SetTexture(0, pTexture);
                RenderMesh(pDevice, g_TeapotMesh);
                g_RenderWhatNext = g_RenderWhat + 1;
                break;
            case 3:
                whatName = "Transparent teapot";
                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, TRUE);
                pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShadeTexture: g_pPFX_PixelShadeTexture);
                pDevice->SetTexture(0, pTexture);
                RenderMesh(pDevice, g_TeapotMesh);
                g_RenderWhatNext = g_RenderWhat + 1;
                break;
            case 4:
                whatName = "Sphere";
                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
                pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShade: g_pPFX_PixelShade);
                pDevice->SetTexture(0, NULL);
                RenderMesh(pDevice, g_SphereMesh);

                g_RenderWhatNext = g_RenderWhat + 1;
                break;
            case 5:
                whatName = "Floor";
                {
                    D3DXMATRIX world;

                    D3DXMatrixIdentity(&world);

                    pDevice->SetTransform(D3DTS_WORLD, &world);

                    SetShaderTransforms(VSConstants, world, view, proj);
                    pDevice->SetVertexShaderConstantF(0, (float const*)VSConstants, 19);
                }

                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
                pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShadeTexture: g_pPFX_PixelShadeTexture);
                pDevice->SetTexture(0, pTexture);
                RenderMesh(pDevice, g_FloorMesh);

                g_RenderWhatNext = 0;
                g_RenderWhichNext = g_RenderWhich + 1;
                break;
        }
    }

    // Render the billboard.

    {
        D3DXMATRIX const world = MatrixScaleYawPitchRollPos(D3DXVECTOR3(4, 3, 1), 3, 0, 0, D3DXVECTOR3(0, 1, 4));

        pDevice->SetTransform(D3DTS_WORLD, &world);

        SetShaderTransforms(VSConstants, world, view, proj);
        pDevice->SetVertexShaderConstantF(0, (float const*)VSConstants, 19);

        pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
        pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShadeTexture: g_pPFX_PixelShadeTexture);
        pDevice->SetTexture(0, pTexture);

        RenderBillboard(pDevice);
    }

    std::string const lightsName = std::string(g_DirectionalLightEnable? "Directional": "") + " " + (g_PointLightEnable? "Point": "") + " " + (g_SpecularLightEnable? "Specular": "");

    return std::string(whatName) + " (" + lightsName + ") | " + (g_UseReferenceShaders? "Reference ": "");
}

std::string RenderBackgroundScene(IDirect3DDevice9* pDevice, IDirect3DTexture9* pTexture, float currentTime)
{
    pDevice->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, D3DXBlack, 1, 0);

    float const cycle = (1 + sin(currentTime * 0.02f)) / 2;
    float const cycle1 = 1 - cycle;

    D3DXVECTOR2 const one (1, 1);

    char const* whatName       = "";
    char const* addressingName = "";
    char const* filterName     = "";

    switch (g_BackgroundAddressing)
    {
    case D3DTADDRESS_CLAMP:
        addressingName = "clamp";
        break;
    case D3DTADDRESS_WRAP:
        addressingName = "wrap";
        break;
    case D3DTADDRESS_MIRROR:
        addressingName = "mirror";
        break;
    }
    switch(g_BackgroundFilter)
    {
    case D3DTEXF_POINT:
        filterName = "point";
        break;
    case D3DTEXF_LINEAR:
        switch(g_BackgroundMipFilter)
        {
        case D3DTEXF_POINT:
            filterName = "bilinear";
            break;
        case D3DTEXF_LINEAR:
            filterName = "trilinear";
            break;
        }
        break;
    }

    pDevice->SetTexture(0, pTexture);
    pDevice->SetSamplerState(0, D3DSAMP_ADDRESSU , g_BackgroundAddressing);
    pDevice->SetSamplerState(0, D3DSAMP_ADDRESSV , g_BackgroundAddressing);
    pDevice->SetSamplerState(0, D3DSAMP_MINFILTER, g_BackgroundFilter);
    pDevice->SetSamplerState(0, D3DSAMP_MAGFILTER, g_BackgroundFilter);
    pDevice->SetSamplerState(0, D3DSAMP_MIPFILTER, g_BackgroundMipFilter);

    switch (g_RenderWhat)
    {
        case 0:
            {
                D3DXVECTOR2 const t0 = one * ((-4) * cycle1 + 0.4f * cycle);
                D3DXVECTOR2 const t1 = one * (  5  * cycle1 + 0.6f * cycle);
                whatName = "Zooming";
                RenderBackground(pDevice, pTexture, t0, t1, false);
                g_RenderWhatNext = g_RenderWhat + 1;
                break;
            }
        case 1:
            {
                D3DXVECTOR2 const t0 = one * (0.00004f * currentTime);
                D3DXVECTOR2 const t1 = one * (0.00004f * currentTime + 1);
                whatName = "Slow panning";
                RenderBackground(pDevice, pTexture, t0, t1, false);
                g_RenderWhatNext = 0;
                g_RenderWhichNext = g_RenderWhich + 1;
                break;
            }
    }

    return std::string(whatName) + " (" + addressingName + " " + filterName + ") ";
}

enum TargetGammaCorrection
{
    TargetGammaWrite,
    TargetGammaManual,
};

struct Configuration
{
    UINT       adapter;
    D3DDEVTYPE devType;
    D3DFORMAT  adapterFormat;

    D3DFORMAT sRGBTextureFormat;
    D3DFORMAT linearTextureFormat;
    D3DFORMAT sRGBTargetFormat;
    D3DFORMAT linearTargetFormat;
};

Configuration g_Config;

D3DFORMAT FindDeviceFormat(IDirect3D9* pD3D, DWORD usage, D3DRESOURCETYPE resType, D3DFORMAT const* fmtList, DWORD numFormats)
{
    for (DWORD i = 0; i < numFormats; ++i)
    {
        D3DFORMAT const format = fmtList[i];

        if (D3D_OK == pD3D->CheckDeviceFormat(g_Config.adapter, g_Config.devType, g_Config.adapterFormat, usage, resType, format))
        {
            return format;
        }
    }
    return D3DFMT_UNKNOWN;
}

template < DWORD numFormats >
__forceinline
D3DFORMAT FindDeviceFormat(IDirect3D9* pD3D, DWORD usage, D3DRESOURCETYPE resType, D3DFORMAT const (&fmtList)[numFormats])
{
    return FindDeviceFormat(pD3D, usage, resType, fmtList, numFormats);
}

IDirect3DTexture9* NewTexture(IDirect3DDevice9* pDevice, char const* fname, bool sRGB)
{
    IDirect3DTexture9* texture = NULL;

    bool const correctGamma = sRGB && g_Config.sRGBTextureFormat == D3DFMT_UNKNOWN;

    D3DFORMAT const format =
        correctGamma? g_Config.linearTextureFormat:
        sRGB?         g_Config.sRGBTextureFormat:
                      D3DFMT_A8R8G8B8;

    HRESULT const hr = D3DXCreateTextureFromFileExA(
        pDevice, fname,
        D3DX_DEFAULT_NONPOW2, D3DX_DEFAULT_NONPOW2, 0,
        0, format, D3DPOOL_MANAGED,
        D3DX_FILTER_POINT | (correctGamma? D3DX_FILTER_SRGB_IN: 0), D3DX_DEFAULT,
        0, NULL, NULL, &texture);

    if (FAILED(hr))
    {
        throw std::string("Failed to load texture ") + fname;
    }

    return texture;
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT ps;
    HDC hdc;

    switch (message)
    {
    case WM_PAINT:
        hdc = BeginPaint(hWnd, &ps);
        EndPaint(hWnd, &ps);
        break;
    case WM_KEYDOWN: 
        switch (wParam) 
        { 
            case VK_ESCAPE:
                PostQuitMessage(0);
                return 0;
            case VK_LEFT: 
                run = false;
                increment -= 1;
                return 0;
            case VK_RIGHT: 
                run = false;
                increment += 1;
                return 0;
            case VK_UP: 
                reset = true;
                return 0;
            case VK_DOWN: 
                run = true;
                return 0;
        }
        return DefWindowProc(hWnd, message, wParam, lParam);
    case WM_CHAR:
        if (g_RenderWhich == 0)
        {
            switch (wParam)
            {
                case 'n':
                case 'N':
                    g_UseAnisotropicLighting = !g_UseAnisotropicLighting;
                    return 0;
                case 'f':
                case 'F':
                    g_FogEnable = !g_FogEnable;
                    return 0;
                case 'l':
                case 'L':
                    g_DirectionalLightEnable = !g_DirectionalLightEnable;
                    return 0;
                case 'p':
                case 'P':
                    g_PointLightEnable = !g_PointLightEnable;
                    return 0;
                case 'c':
                case 'C':
                    g_SpecularLightEnable = !g_SpecularLightEnable;
                    return 0;
                case 'r':
                case 'R':
                    g_UseReferenceShaders = !g_UseReferenceShaders;
                    return 0;
            }
        }
        else if (g_RenderWhich == 1)
        {
            switch (wParam)
            {
                case 'a':
                case 'A':
                    g_BackgroundAddressingIndex = (g_BackgroundAddressingIndex + 1) % 3;
                    g_BackgroundAddressing = g_BackgroundAddressingList[g_BackgroundAddressingIndex];
                    return 0;
                case 'f':
                case 'F':
                    g_BackgroundFilterIndex = (g_BackgroundFilterIndex + 1) % 3;
                    g_BackgroundFilter    = g_BackgroundFilterList[g_BackgroundFilterIndex][0];
                    g_BackgroundMipFilter = g_BackgroundFilterList[g_BackgroundFilterIndex][1];
                    return 0;
            }
        }

        switch (wParam)
        {
            case 'g':
            case 'G':
                g_UseGammaCorrection = !g_UseGammaCorrection;
                return 0;
            case 't':
            case 'T':
                g_UseGridTexture = !g_UseGridTexture;
                return 0;
            case ' ':
                g_RenderWhat  = g_RenderWhatNext;
                g_RenderWhich = g_RenderWhichNext;
                return 0;
        }
        return DefWindowProc(hWnd, message, wParam, lParam);
    case WM_DESTROY:
        PostQuitMessage(0);
        break;
    default:
        return DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}

int APIENTRY MyWinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPWSTR    lpCmdLine,
                     int       nCmdShow)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    {
        WNDCLASSEX wcex;

        wcex.cbSize = sizeof(WNDCLASSEX);

        wcex.style			= CS_HREDRAW | CS_VREDRAW;
        wcex.lpfnWndProc	= WndProc;
        wcex.cbClsExtra		= 0;
        wcex.cbWndExtra		= 0;
        wcex.hInstance		= hInstance;
        wcex.hIcon			= LoadIcon(NULL, IDI_APPLICATION);
        wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
        wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);
        wcex.lpszMenuName	= NULL;
        wcex.lpszClassName	= L"SWRenderer";
        wcex.hIconSm		= LoadIcon(NULL, IDI_APPLICATION);

        RegisterClassEx(&wcex);
    }

    HWND const hWnd = CreateWindow(
        L"SWRenderer", L"Software Renderer",
        WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT, 0, 640, 480,
        NULL, NULL, hInstance, NULL);

    if (!hWnd)
    {
        throw "Failed to create the window";
    }

    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);

    D3DPRESENT_PARAMETERS pp;

    {
        RECT rect;
        GetClientRect(hWnd, &rect);

        DWORD const width  = rect.right - rect.left;
        DWORD const height = rect.bottom - rect.top;

        memset(&pp, 0, sizeof(pp));
        pp.BackBufferWidth            = width;
        pp.BackBufferHeight           = height;
        pp.BackBufferFormat           = D3DFMT_UNKNOWN;
        pp.BackBufferCount            = 1;
        pp.MultiSampleType            = D3DMULTISAMPLE_NONE;
        pp.SwapEffect                 = D3DSWAPEFFECT_DISCARD;
        pp.hDeviceWindow              = hWnd;
        pp.Windowed                   = TRUE;
        pp.EnableAutoDepthStencil     = TRUE;
        pp.AutoDepthStencilFormat     = D3DFMT_D24S8;
        pp.Flags                      = D3DPRESENTFLAG_DISCARD_DEPTHSTENCIL;
        pp.FullScreen_RefreshRateInHz = 0;
        pp.PresentationInterval       = D3DPRESENT_INTERVAL_ONE;
    }

    IDirect3DDevice9* pDevice = NULL;

    {
        IDirect3D9* const pD3D = Direct3DCreate9(D3D_SDK_VERSION);
        if (!pD3D)
        {
            throw "Failed to initialize the D3D interface";
        }

        HRESULT hr = pD3D->CreateDevice(
            D3DADAPTER_DEFAULT,
            D3DDEVTYPE_HAL,
            hWnd,
            D3DCREATE_MIXED_VERTEXPROCESSING,
            &pp,
            &pDevice);

        if (FAILED(hr))
        {
            hr = pD3D->CreateDevice(
                D3DADAPTER_DEFAULT,
                D3DDEVTYPE_HAL,
                hWnd,
                D3DCREATE_SOFTWARE_VERTEXPROCESSING,
                &pp,
                &pDevice);
        }

        if (FAILED(hr))
        {
            throw "Failed to initialize the D3D device";
        }

        g_Config.adapter       = D3DADAPTER_DEFAULT;
        g_Config.devType       = D3DDEVTYPE_HAL;
        g_Config.adapterFormat = pp.BackBufferFormat;

        D3DFORMAT const sRGBTextureFormats[] =
        {
            D3DFMT_A8R8G8B8,
        };
        g_Config.sRGBTextureFormat = FindDeviceFormat(pD3D, D3DUSAGE_QUERY_SRGBREAD, D3DRTYPE_TEXTURE, sRGBTextureFormats);
        //g_Config.sRGBTextureFormat = D3DFMT_UNKNOWN;

        D3DFORMAT const linearTextureFormats[] =
        {
            D3DFMT_A16B16G16R16,
            D3DFMT_A2R10G10B10,
            D3DFMT_A2B10G10R10,
            D3DFMT_A8R8G8B8,
        };
        g_Config.linearTextureFormat = FindDeviceFormat(pD3D, 0, D3DRTYPE_TEXTURE, linearTextureFormats);

        DWORD const targetUsage = D3DUSAGE_RENDERTARGET|D3DUSAGE_QUERY_POSTPIXELSHADER_BLENDING;

        D3DFORMAT const sRGBTargetFormats[] =
        {
            D3DFMT_A8R8G8B8,
        };
        g_Config.sRGBTargetFormat = FindDeviceFormat(pD3D, targetUsage|D3DUSAGE_QUERY_SRGBWRITE, D3DRTYPE_TEXTURE, sRGBTargetFormats);
        //g_Config.sRGBTargetFormat = D3DFMT_UNKNOWN;

        D3DFORMAT const linearTargetFormats[] =
        {
            D3DFMT_A16B16G16R16,
            D3DFMT_A2R10G10B10,
            D3DFMT_A2B10G10R10,
            D3DFMT_A8R8G8B8,
        };
        g_Config.linearTargetFormat = FindDeviceFormat(pD3D, targetUsage, D3DRTYPE_TEXTURE, linearTargetFormats);

        pD3D->Release();
    }

    HWRenderer::Init(pDevice);
    InitShaders(pDevice);

    IDirectInputDevice8* pMouse = NULL;
    IDirectInputDevice8* pKeyboard = NULL;

    {
        IDirectInput8* pDInput = NULL;
        HRESULT hr = DirectInput8Create(hInstance, DIRECTINPUT_VERSION, IID_IDirectInput8, (LPVOID*)&pDInput, NULL);

        if (FAILED(hr))
        {
            throw "Failed to initialize DirectInput";
        }

        hr = pDInput->CreateDevice(GUID_SysMouse, &pMouse, NULL);

        if (FAILED(hr))
        {
            throw "Failed to initialize the mouse device";
        }

        hr = pDInput->CreateDevice(GUID_SysKeyboard, &pKeyboard, NULL);

        if (FAILED(hr))
        {
            throw "Failed to initialize the keyboard device";
        }

        pDInput->Release();

        pMouse->SetDataFormat(&c_dfDIMouse);
        pMouse->SetCooperativeLevel(hWnd, DISCL_EXCLUSIVE | DISCL_FOREGROUND);
        pMouse->Acquire();

        pKeyboard->SetDataFormat(&c_dfDIKeyboard);
        pKeyboard->SetCooperativeLevel(hWnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);
        pKeyboard->Acquire();
    }

    unsigned int const NumBackBuffers = 2;
    unsigned int const BackBufferWidth  = 320;
    unsigned int const BackBufferHeight = 240;
    D3DFORMAT const    BackBufferFormat = g_Config.sRGBTargetFormat != D3DFMT_UNKNOWN? g_Config.sRGBTargetFormat: g_Config.linearTargetFormat;

    IDirect3DTexture9* backBuffers[NumBackBuffers] = { NULL, NULL };
    unsigned int currentBackBuffer = 0;

    for (unsigned int i = 0; i < NumBackBuffers; ++i)
    {
        HRESULT const hr = pDevice->CreateTexture(
            BackBufferWidth, BackBufferHeight, 1,
            D3DUSAGE_RENDERTARGET, BackBufferFormat, D3DPOOL_DEFAULT,
            &backBuffers[i], NULL);

        if (FAILED(hr))
        {
            throw "Failed to create back buffer texture";
        }
    }

    IDirect3DTexture9* const pTextureGamma  = NewTexture(pDevice, "Texture.jpg", false);
    IDirect3DTexture9* const pTextureLinear = NewTexture(pDevice, "Texture.jpg", true);
    IDirect3DTexture9* const pGridGamma     = NewTexture(pDevice, "Grid.jpg", false);
    IDirect3DTexture9* const pGridLinear    = NewTexture(pDevice, "Grid.jpg", true);
    IDirect3DTexture9* const pHeightMap     = NewTexture(pDevice, "HeightMap.tga", false);

    g_GroundMesh = CreateGridMesh(pDevice, pHeightMap, 100, 100, 1, 5, 1, D3DXVECTOR3(0, -4, 5), D3DXWhite, D3DXBlack);
    g_FloorMesh  = CreateGridMesh(pDevice, pHeightMap,  20,  20, 5, 0, 5, D3DXVECTOR3(0,  1, 5), D3DXWhite, D3DXBlack);

    {
        ID3DXMesh* pMesh = NULL;

        D3DXCreateTeapot(pDevice, &pMesh, NULL);

        g_TeapotMesh = ConvertMesh(pDevice, pMesh);

        pMesh->Release();

        D3DXCreateSphere(pDevice, 1, 100, 50, &pMesh, NULL);

        g_SphereMesh = ConvertMesh(pDevice, pMesh, D3DXCOLOR(1, 0.5f, 0.2f, 1));

        pMesh->Release();
    }

    // Main message loop:
    MSG msg = { 0 };

    Averager<DWORD, 32> timeAverager;
    DWORD startTime = timeGetTime();
    DWORD frameCount = 0;
    char framerateMsg[256] = "";

    float currentTime = 0;

    while (msg.message != WM_QUIT)
    {
        if (reset)
        {
            currentTime = 0;
            reset = false;
        }
        else if (run)
        {
            currentTime += 1;
        }
        else if (increment != 0)
        {
            currentTime += increment;
            increment = 0;
        }

        {
            DWORD const time = timeGetTime();

            timeAverager.AddSample(time - startTime);

            ++frameCount;
            if (frameCount % 32 == 0)
            {
                sprintf_s(framerateMsg, "%8.2f FPS", 1000.0 / timeAverager.GetAverage());
                startTime = timeGetTime();
            }
            else
            {
                startTime = time;
            }
        }

        // If the window size has changed, reset the device.

        {
            RECT rect;
            GetClientRect(hWnd, &rect);

            DWORD const width  = rect.right - rect.left;
            DWORD const height = rect.bottom - rect.top;
            if (width != pp.BackBufferWidth || height != pp.BackBufferHeight)
            {
                // Release all non-managed resources.

                for (unsigned int i = 0; i < NumBackBuffers; ++i)
                {
                    backBuffers[i]->Release();
                    backBuffers[i] = NULL;
                }

                // Change D3D's back buffer size.

                pp.BackBufferWidth  = width;
                pp.BackBufferHeight = height;
                pDevice->Reset(&pp);

                // Recreate all non-managed resources.

                for (unsigned int i = 0; i < NumBackBuffers; ++i)
                {
                    HRESULT const hr = pDevice->CreateTexture(
                        BackBufferWidth, BackBufferHeight, 1,
                        D3DUSAGE_RENDERTARGET, BackBufferFormat, D3DPOOL_DEFAULT,
                        &backBuffers[i], NULL);

                    if (FAILED(hr))
                    {
                        throw "Failed to create back buffer texture";
                    }
                }
            }
        }

        // Render.

        {
            pDevice->BeginScene();

            IDirect3DTexture9* const pTexture =
                g_UseGridTexture?
                    (g_UseGammaCorrection? pGridLinear   : pGridGamma):
                    (g_UseGammaCorrection? pTextureLinear: pTextureGamma);

            pDevice->SetSamplerState(0, D3DSAMP_SRGBTEXTURE, g_UseGammaCorrection && g_Config.sRGBTextureFormat != D3DFMT_UNKNOWN);
            pDevice->SetRenderState (D3DRS_SRGBWRITEENABLE,  g_UseGammaCorrection && g_Config.sRGBTargetFormat  != D3DFMT_UNKNOWN);

            pDevice->SetPixelShader(g_UseReferenceShaders? g_pPFX_ReferencePixelShade: g_pPFX_PixelShade);

            IDirect3DTexture9* const pTargetTexture = backBuffers[currentBackBuffer];
            currentBackBuffer = (currentBackBuffer + 1) % NumBackBuffers;

            IDirect3DSurface9* pOldBackBuffer = NULL;
            pDevice->GetRenderTarget(0, &pOldBackBuffer);

            {
                IDirect3DSurface9* pTargetSurface = NULL;
                pTargetTexture->GetSurfaceLevel(0, &pTargetSurface);

                pDevice->SetRenderTarget(0, pTargetSurface);

                pTargetSurface->Release();
            }

            std::string whatName;

            // SW-Render a scene into the back buffer texture.

            switch (g_RenderWhich)
            {
            default:
                g_RenderWhich = 0;
            case 0:
                whatName = RenderScene(pDevice, pTexture, currentTime);
                break;
            case 1:
                whatName = RenderBackgroundScene(pDevice, pTexture, currentTime);
                break;
            }

            // Update the window title.

            {
                char const* const gammaName = g_UseGammaCorrection? "Gamma-correct": "Not Gamma-correct";

                SetWindowTextA(hWnd, (std::string(whatName) + " | " + gammaName + " | " + framerateMsg).c_str());
            }

/*
            RenderFunc(pDevice, &Gamma22         , D3DXCOLOR(1, 0, 0, 1));
            RenderFunc(pDevice, &Gamma22Inv      , D3DXCOLOR(0, 1, 1, 1));
            RenderFunc(pDevice, &Gamma22Approx   , D3DXCOLOR(0, 1, 0, 1));
            RenderFunc(pDevice, &Gamma22InvApprox, D3DXCOLOR(1, 0, 1, 1));
*/

            pDevice->SetRenderTarget(0, pOldBackBuffer);
            pOldBackBuffer->Release();

            // Present the back buffer texture onto the window.

            {
                struct CUSTOMVERTEX
                {
                    FLOAT x, y, z, rhw; // The transformed position for the vertex.
                    float u, v;         // The vertex color.
                };

                DWORD const D3DFVF_CUSTOMVERTEX = D3DFVF_XYZRHW | D3DFVF_TEX1;

                float const width  = float(pp.BackBufferWidth);
                float const height = float(pp.BackBufferHeight);

                CUSTOMVERTEX const vertices[] =
                {
                    { 0.0f,  0.0f,   0.5f, 1.0f, 0, 0, }, // x, y, z, rhw, u, v
                    { width, 0.0f,   0.5f, 1.0f, 1, 0, },
                    { 0.0f,  height, 0.5f, 1.0f, 0, 1, },
                    { width, height, 0.5f, 1.0f, 1, 1, },
                };

                pDevice->SetVertexShader(NULL);
                pDevice->SetFVF(D3DFVF_CUSTOMVERTEX);
                pDevice->SetTexture(0, pTargetTexture);
                pDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_FALSE);
                pDevice->SetRenderState(D3DRS_ZWRITEENABLE, FALSE);
                pDevice->SetRenderState(D3DRS_FOGENABLE, FALSE);
                pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
                pDevice->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_POINT);
                pDevice->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_POINT);
                pDevice->SetSamplerState(0, D3DSAMP_ADDRESSU, D3DTADDRESS_CLAMP);
                pDevice->SetSamplerState(0, D3DSAMP_ADDRESSV, D3DTADDRESS_CLAMP);

                pDevice->SetSamplerState(0, D3DSAMP_SRGBTEXTURE, FALSE);
                pDevice->SetRenderState (D3DRS_SRGBWRITEENABLE, FALSE);

                if (g_Config.sRGBTargetFormat == D3DFMT_UNKNOWN && g_UseGammaCorrection)
                {
                    // The texture is in linear space.
                    // We use a pixel shader to apply the gamma conversion.

                    pDevice->SetPixelShader(g_pPFX_LinearToGamma);
                }
                else
                {
                    // The texture has already been re-gammaed.

                    pDevice->SetPixelShader(NULL);
                }

                pDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, vertices, sizeof(vertices[0]));

                pDevice->SetTexture(0, NULL);
            }

            pDevice->EndScene();

            pDevice->Present(NULL, NULL, NULL, NULL);
        }

        // Poll the mouse

        {
            DIMOUSESTATE state;
            HRESULT hr = pMouse->GetDeviceState(sizeof(state), &state);

            if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED)
            {
                pMouse->Acquire();
            }
            else
            {
                g_CameraYaw   += float(state.lX) * 0.001f;
                g_CameraPitch -= float(state.lY) * 0.001f;
            }
        }

        // Poll the keyboard

        {
            BYTE state[256];
            HRESULT hr = pKeyboard->GetDeviceState(sizeof(state), &state);

            if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED)
            {
                pKeyboard->Acquire();
            }
            else
            {
                if (g_RenderWhich == 0)
                {
                    if (state[DIK_A] & 0x80)
                    {
                        D3DXVECTOR3 const inc(-g_Speed * cos(g_CameraYaw), 0, g_Speed * sin(g_CameraYaw));
                        g_CameraEye += inc;
                    }
                    if (state[DIK_S] & 0x80)
                    {
                        D3DXVECTOR3 const inc(-g_Speed * sin(g_CameraYaw), 0, -g_Speed * cos(g_CameraYaw));
                        g_CameraEye += inc;
                    }
                    if (state[DIK_D] & 0x80)
                    {
                        D3DXVECTOR3 const inc(g_Speed * cos(g_CameraYaw), 0, -g_Speed * sin(g_CameraYaw));
                        g_CameraEye += inc;
                    }
                    if (state[DIK_W] & 0x80)
                    {
                        D3DXVECTOR3 const inc(g_Speed * sin(g_CameraYaw), 0, g_Speed * cos(g_CameraYaw));
                        g_CameraEye += inc;
                    }
                    if (state[DIK_Q] & 0x80)
                    {
                        g_CameraEye.y += g_Speed;
                    }
                    if (state[DIK_E] & 0x80)
                    {
                        g_CameraEye.y -= g_Speed;
                    }
                }
            }
        }

        // Pump windows messages.

        while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) && msg.message != WM_QUIT)
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    }

    for (unsigned int i = 0; i < NumBackBuffers; ++i)
    {
        backBuffers[i]->Release();
    }

    pDevice->Release();

    return (int) msg.wParam;
}

int APIENTRY wWinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPWSTR    lpCmdLine,
                     int       nCmdShow)
{
    int exitCode;

    try
    {
        exitCode = MyWinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow);
    }
    catch (char const* msg)
    {
        OutputDebugStringA(msg);
        MessageBoxA(NULL, msg, "Detected error", MB_OK);
        exitCode = 1;
    }
    catch (std::string msg)
    {
        OutputDebugStringA(msg.c_str());
        MessageBoxA(NULL, msg.c_str(), "Detected error", MB_OK);
        exitCode = 1;
    }

    return exitCode;
}
